<?php

declare(strict_types=1);

namespace Gls\GlsPoland\PrestaShop\Hook;

use PrestaShop\PrestaShop\Adapter\Hook\HookDispatcher;
use PrestaShop\PrestaShop\Core\Hook\HookDispatcherInterface;
use PrestaShop\PrestaShop\Core\Hook\HookInterface;
use PrestaShop\PrestaShop\Core\Hook\RenderedHookInterface;
use PrestaShopBundle\Service\Hook\HookEvent;
use PrestaShopBundle\Service\Hook\RenderingHookEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * @internal
 *
 * Fixes the output of recursively called "display" hooks. The default implementation stores content returned by modules
 * in a class property, resetting its value after all listeners are called but without checking if it is empty when
 * dispatching the current event. As a result, when calling a hook from another hook (e.g. rendering grid templates in
 * admin order details page hooks), the previously rendered content is prepended to the hook at the top of the call
 * stack, while only content from the listeners called after the recursive {@see self::dispatch()} call is displayed
 * for the original hook.
 *
 * @see HookDispatcher::doDispatch()
 *
 * The decorated class has to be extended since at least {@see \PrestaShop\PrestaShop\Core\Hook\HookDispatcher} relies
 * on implementation.
 */
final class RecursionHandlingHookDispatcher extends HookDispatcher
{
    /**
     * @var HookDispatcherInterface
     */
    private $dispatcher;

    /**
     * @var \ReflectionProperty|false
     */
    private $renderingContent;

    /**
     * @param HookDispatcher $dispatcher
     */
    public function __construct(HookDispatcherInterface $dispatcher)
    {
        $this->dispatcher = $dispatcher;
    }

    public function __call(string $name, array $arguments)
    {
        return $this->dispatcher->$name(...$arguments);
    }

    public function dispatch($eventName, $event = null): object
    {
        if (!$event instanceof RenderingHookEvent) {
            return $this->dispatcher->dispatch($eventName, $event);
        }

        return $this->callHandlingPreviousHookContent(function () use ($eventName, $event) {
            return $this->dispatcher->dispatch($eventName, $event);
        });
    }

    public function renderForParameters($eventName, array $parameters = []): RenderingHookEvent
    {
        return $this->callHandlingPreviousHookContent(function () use ($eventName, $parameters) {
            return $this->dispatcher->renderForParameters($eventName, $parameters);
        });
    }

    public function dispatchMultiple(array $eventNames, array $eventParameters): void
    {
        $this->dispatcher->dispatchMultiple($eventNames, $eventParameters);
    }

    public function dispatchForParameters($eventName, array $parameters = []): HookEvent
    {
        return $this->dispatcher->dispatchForParameters($eventName, $parameters);
    }

    public function addListener($eventName, $listener, $priority = 0): void
    {
        $this->dispatcher->addListener($eventName, $listener, $priority);
    }

    public function addSubscriber(EventSubscriberInterface $subscriber): void
    {
        $this->dispatcher->addSubscriber($subscriber);
    }

    public function removeListener($eventName, $listener): void
    {
        $this->dispatcher->removeListener($eventName, $listener);
    }

    public function removeSubscriber(EventSubscriberInterface $subscriber): void
    {
        $this->dispatcher->removeSubscriber($subscriber);
    }

    public function getListeners($eventName = null): array
    {
        return $this->dispatcher->getListeners($eventName);
    }

    public function getListenerPriority($eventName, $listener): ?int
    {
        return $this->dispatcher->getListenerPriority($eventName, $listener);
    }

    public function hasListeners($eventName = null): bool
    {
        return $this->dispatcher->hasListeners($eventName);
    }

    public function dispatchHook(HookInterface $hook): HookEvent
    {
        return $this->dispatcher->dispatchHook($hook);
    }

    public function dispatchWithParameters($hookName, array $hookParameters = []): void
    {
        $this->dispatcher->dispatchWithParameters($hookName, $hookParameters);
    }

    public function dispatchRendering(HookInterface $hook): RenderedHookInterface
    {
        return $this->callHandlingPreviousHookContent(function () use ($hook) {
            return $this->dispatcher->dispatchRendering($hook);
        });
    }

    public function dispatchRenderingWithParameters($hookName, array $hookParameters = []): RenderedHookInterface
    {
        return $this->callHandlingPreviousHookContent(function () use ($hookName, $hookParameters) {
            return $this->dispatcher->dispatchRenderingWithParameters($hookName, $hookParameters);
        });
    }

    /**
     * @template T of (RenderingHookEvent|RenderedHookInterface)
     *
     * @param callable(): T $callback
     *
     * @return T
     */
    private function callHandlingPreviousHookContent(callable $callback)
    {
        if (null === $renderingContent = $this->getRenderingContent()) {
            return $callback();
        }

        if ([] === $previousContent = $renderingContent->getValue($this->dispatcher)) {
            return $callback();
        }

        $renderingContent->setValue($this->dispatcher, []);
        $result = $callback();
        $renderingContent->setValue($this->dispatcher, $previousContent);

        return $result;
    }

    private function getRenderingContent(): ?\ReflectionProperty
    {
        if (isset($this->renderingContent)) {
            return $this->renderingContent ?: null;
        }

        $class = new \ReflectionClass($this->dispatcher);
        if (!$class->hasProperty('renderingContent')) {
            $this->renderingContent = false;

            return null;
        }

        $this->renderingContent = $class->getProperty('renderingContent');
        $this->renderingContent->setAccessible(true);

        return $this->renderingContent;
    }
}
