<?php

declare(strict_types=1);

namespace Gls\GlsPoland\PrestaShop\Checkout;

use Gls\GlsPoland\Entity\Carrier;
use Gls\GlsPoland\PrestaShop\Checkout\Presenter\CheckoutConfigPresenter;
use Gls\GlsPoland\PrestaShop\Helper\CarrierFinder;
use Gls\GlsPoland\PrestaShop\Templating\RendererInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

abstract class AbstractCheckout implements CheckoutInterface
{
    use CarriersByDeliveryOptionFinderTrait;
    use CheckoutTrait;

    public const MAP_SDK_JS_URI = 'https://mapa.gls-poland.com/js/v4.0/maps_sdk.js';
    /**
     * @deprecated since 1.1.0
     */
    public const MAP_SDK_CSS_URI = 'https://mapa.szybkapaczka.pl/css/v3.1/maps_sdk.css';

    private $validator;
    private $persister;

    public function __construct(RendererInterface $renderer, CheckoutConfigPresenter $dataPresenter, CarrierFinder $carrierFinder, DeliveryOptionValidatorInterface $validator, DeliveryOptionDataPersisterInterface $persister)
    {
        $this->renderer = $renderer;
        $this->dataPresenter = $dataPresenter;
        $this->carrierFinder = $carrierFinder;
        $this->validator = $validator;
        $this->persister = $persister;
    }

    protected function validateDeliveryOption(\Cart $cart, array $requestParameters): bool
    {
        if (!isset($requestParameters['delivery_option']) || !is_array($requestParameters['delivery_option'])) {
            $this->addError($this->getTranslator()->trans('Malformed request.', [], 'Modules.Glspoland.Checkout'));

            return false;
        }

        if ([] === $carriers = $this->getCarriersByDeliveryOption($cart, $requestParameters['delivery_option'])) {
            return true;
        }

        foreach ($carriers as $carrierId => $carrier) {
            if (!$this->validateCarrierData($carrier, $requestParameters['gls_poland'][$carrierId] ?? null)) {
                return false;
            }
        }

        return true;
    }

    protected function saveDeliveryOption(\Cart $cart, Request $request): void
    {
        if (!$request->request->has('delivery_option')) {
            return;
        }

        $carriers = $this->getCarriersByDeliveryOption($cart, $request->request->get('delivery_option'));
        $data = $request->request->get('gls_poland', []);

        $this->persister->save($cart, $carriers, $data);
    }

    private function validateCarrierData(Carrier $carrier, $data): bool
    {
        try {
            $violations = $this->validator->validate($carrier, $data);
        } catch (UnexpectedTypeException $e) {
            $this->addError($this->getTranslator()->trans('Malformed request.', [], 'Modules.Glspoland.Checkout'));

            return false;
        }

        if (0 === count($violations)) {
            return true;
        }

        foreach ($violations as $violation) {
            $this->addError($violation->getMessage());
        }

        return false;
    }
}
