<?php

declare(strict_types=1);

namespace Gls\GlsPoland\Entity;

use Doctrine\ORM\Mapping as ORM;
use Gls\GlsPoland\Carrier\CarrierId;
use Gls\GlsPoland\Carrier\TimeDefiniteService;
use Gls\GlsPoland\Country\IsoCode;

/**
 * @ORM\Entity()
 */
class CourierCarrier extends Carrier
{
    /**
     * @ORM\Column(type="boolean")
     */
    private $cashOnDelivery;

    /**
     * @ORM\Column(type="boolean")
     */
    private $saturdayDelivery;

    /**
     * @ORM\Column(type="boolean", name="delivery_before_10")
     */
    private $deliveryBefore10;

    /**
     * @ORM\Column(type="boolean", name="delivery_before_12")
     */
    private $deliveryBefore12;

    /**
     * @ORM\Column(type="boolean")
     */
    private $documentReturn;

    /**
     * @param IsoCode[] $countryIsoCodes
     */
    public function __construct(
        CarrierId $id,
        array $countryIsoCodes,
        bool $isCashOnDelivery = false,
        bool $isSaturdayDelivery = false,
        bool $isDeliveryBefore10 = false,
        bool $isDeliveryBefore12 = false,
        bool $isDocumentReturn = false
    ) {
        if ($isDeliveryBefore10 && $isDeliveryBefore12) {
            throw new \LogicException('Delivery before 10:00 AM and before 12:00 AM services are exclusive.');
        }

        parent::__construct($id, ...$countryIsoCodes);

        $this->cashOnDelivery = $isCashOnDelivery;
        $this->saturdayDelivery = $isSaturdayDelivery;
        $this->deliveryBefore10 = $isDeliveryBefore10;
        $this->deliveryBefore12 = $isDeliveryBefore12;
        $this->documentReturn = $isDocumentReturn;
    }

    public function isCashOnDelivery(): bool
    {
        return $this->cashOnDelivery;
    }

    public function isSaturdayDelivery(): bool
    {
        return $this->saturdayDelivery;
    }

    public function getTimeDefiniteService(): ?TimeDefiniteService
    {
        if ($this->deliveryBefore10) {
            return TimeDefiniteService::from(TimeDefiniteService::DELIVERY_BEFORE_10);
        }

        if ($this->deliveryBefore12) {
            return TimeDefiniteService::from(TimeDefiniteService::DELIVERY_BEFORE_12);
        }

        return null;
    }

    public function isDeliveryBefore10(): bool
    {
        return $this->deliveryBefore10;
    }

    public function isDeliveryBefore12(): bool
    {
        return $this->deliveryBefore12;
    }

    public function isDocumentReturn(): bool
    {
        return $this->documentReturn;
    }

    public function isExpressDelivery(): bool
    {
        return $this->isSaturdayDelivery()
            || $this->isDeliveryBefore10()
            || $this->isDeliveryBefore12();
    }
}
