<?php

declare(strict_types=1);

namespace Gls\GlsPoland\Controller\Front;

use Gls\GlsPoland\PrestaShop\Checkout\Exception\ValidationException;
use Gls\GlsPoland\PrestaShop\Checkout\Message\UpdateParcelShopChoiceCommand;
use PrestaShop\PrestaShop\Core\CommandBus\CommandBusInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\ConflictHttpException;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;

final class CheckoutController
{
    /**
     * @var \Context
     */
    private $context;

    /**
     * @var CommandBusInterface
     */
    private $bus;

    public function __construct(\Context $context, CommandBusInterface $bus)
    {
        $this->context = $context;
        $this->bus = $bus;
    }

    public function selectParcelShop(Request $request): Response
    {
        if (!$this->context->cart->hasProducts()) {
            throw new ConflictHttpException('Cart is empty.');
        }

        $data = $this->decodeRequest($request);

        if (!isset($data['parcelShopId'])) {
            throw new BadRequestHttpException('Required parameter "parcelShopId" is missing.');
        }

        $command = new UpdateParcelShopChoiceCommand($this->context->cart, (string) $data['parcelShopId']);

        try {
            $this->bus->handle($command);
        } catch (ValidationException $e) {
            throw new UnprocessableEntityHttpException($e->getMessage(), $e);
        }

        return JsonResponse::create(null, Response::HTTP_NO_CONTENT)->setContent(null);
    }

    private function decodeRequest(Request $request): array
    {
        $data = json_decode($request->getContent(), true);

        if (null === $data && JSON_ERROR_NONE !== json_last_error()) {
            throw new BadRequestHttpException('Could not decode request as JSON.');
        }

        if (!is_array($data)) {
            throw new BadRequestHttpException('Expected the request to decode into an array.');
        }

        return $data;
    }
}
